/***********************************************************************************************************************
MMBasic

PWM.c

Handles the PWM and SERVO commands

Copyright 2011 - 2022 Geoff Graham.  All Rights Reserved.

This file and modified versions of this file are supplied to specific individuals or organisations under the following
provisions:

- This file, or any files that comprise the MMBasic source (modified or not), may not be distributed or copied to any other
  person or organisation without written permission.

- Object files (.o and .hex files) generated using this file (modified or not) may not be distributed or copied to any other
  person or organisation without written permission.

- This file is provided in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

************************************************************************************************************************/

#include "../../Version.h"


static unsigned int p2 = -1, p3 = -1;                               // a place to save the current prescales for the timers
static char oc1, oc2, oc3, oc4, oc5;                                // flags which are true if the OCx module is opened

// the PWM and SERVO commands use the same function
void cmd_pwm(void) {
    int i, j, channel, per, f, prescale, ChangeOK = 0;
    int dcy[3]= {-1, -1, -1};

    getargs(&cmdline, 9, ",");
    if((argc & 0x01) == 0 || argc < 3) error("Argument count");

    channel = getint(argv[0], 1, 2) - 1;

#ifdef MX470
    if((!(CurrentlyPlaying == P_NOTHING || CurrentlyPlaying == P_STOPPED)) && channel == 1) error("In use for audio");
#endif

    if(checkstring(argv[2], "STOP")){
        PWMClose(channel);
        return;
    }

    if(cmdtoken == GetCommandValue("PWM")) {
        f = getint(argv[2], 20, 500000);
        if(argc < 5) error("Argument count");
        if(f < 1250) {
            per = ((BusSpeed/f) / 64) -1;
            prescale = channel ? T2_PS_1_64:T3_PS_1_64;
        } else {
            per = (BusSpeed/f) - 1;
            prescale = channel ? T2_PS_1_1:T3_PS_1_1;
        }
        for(i = 0; i < (argc - 3)/2; i++) {
            MMFLOAT duty = getnumber(argv[(i * 2) + 4]);
            int dty = getinteger(argv[(i * 2) + 4]);
            if(dty < 0)
                dcy[i] = -dty;
            else {
                if(duty < 0.0 || duty > 100.0) error("Number out of bounds");
                dcy[i] = duty * 100.0;
                if(duty == 100.0) dcy[i] = 10100;                       // 101%
                dcy[i] = (per * dcy[i]) / 10000;
                if(ChangeOK < dcy[i]) ChangeOK = dcy[i];
            }
        }
    } else {
        // Command must be SERVO
        f = getinteger(argv[2]);
        if(f >= 20) {                                               //must be a frequency
            if(f > 1000) error("% out of bounds", f);
            j = 4;
        } else {
            f = 50;
            j = 2;
        }

        per = ((BusSpeed/ f) / 64) - 1;
        prescale = channel ? T2_PS_1_64:T3_PS_1_64;

        for(i = 0; i < (argc - j + 1)/2; i++) {
            MMFLOAT ontime = getnumber(argv[(i * 2) + j]);
            if(ontime < 0.01 || ontime > 18.9) error("Number out of bounds");
            dcy[i] = (((MMFLOAT)BusSpeed/6.4) * ontime)/10000 ;
            if(ChangeOK < dcy[i]) ChangeOK = dcy[i];
        }
    }

    if(dcy[0] < 0) error("Argument count");

    // this is channel 1
    if(channel == 0) {

        if(p2 == T2_PS_1_1) INTDisableInterrupts();
        if(p2 != prescale) {                                        // if we are opening the channel or changing the prescaler
            if(T2CONbits.ON) while(TMR2 < PR2 - 64);                // if the timer is running wait until it is near the end of its period
            OpenTimer2(T2_ON | prescale, per);
            p2 = prescale;
        } else if(PR2 != per) {                                     // if we are just changing the frequency
            while(TMR2 > per - 24);                                 // wait for it to be safe to do so
            PR2 = per;
        }
        INTEnableInterrupts();

        // output 1A
        if(!oc3) {
            ExtCfg(PWM_CH1_PIN, EXT_DIG_OUT, 0);                    // this is the first time so initialise everything
            ExtCfg(PWM_CH1_PIN, EXT_COM_RESERVED, 0);
            PWM_CH1_OPEN;
            oc3 = true;
            OC3CON = 0;                                             // reset the OC module so that we can set up again correctly
            OpenOC3( OC_ON | OC_TIMER_MODE16 | OC_TIMER2_SRC | OC_PWM_FAULT_PIN_DISABLE , dcy[0], dcy[0] );
        } else
            OC3RS = dcy[0];                                         // this is output 1A duty cycle

        // output 1B
        if(dcy[1] >= 0) {
            if(!oc2) {
                ExtCfg(PWM_CH2_PIN, EXT_DIG_OUT, 0);                // this is the first time so initialise everything
                ExtCfg(PWM_CH2_PIN, EXT_COM_RESERVED, 0);
                PWM_CH2_OPEN;
                oc2 = true;
                OC2CON = 0;                                         // reset the OC module so that we can set up again correctly
                OpenOC2( OC_ON | OC_TIMER_MODE16 | OC_TIMER2_SRC | OC_PWM_FAULT_PIN_DISABLE , dcy[1], dcy[1] );
            } else
                OC2RS = dcy[1];                                     // this is output 1B duty cycle
        }

        // output 1C
        if(dcy[2] >= 0) {
            if(!oc5) {
                ExtCfg(PWM_CH3_PIN, EXT_DIG_OUT, 0);                // this is the first time so initialise everything
                ExtCfg(PWM_CH3_PIN, EXT_COM_RESERVED, 0);
                PWM_CH3_OPEN;
                oc5 = true;
                OC5CON = 0;                                         // reset the OC module so that we can set up again correctly
                OpenOC5( OC_ON | OC_TIMER_MODE16 | OC_TIMER2_SRC | OC_PWM_FAULT_PIN_DISABLE , dcy[2], dcy[2] );
            } else
                OC5RS = dcy[2];                                     // this is output 1C duty cycle
        }

    } else {
        // this is channel 2

        if(p2 == T3_PS_1_1) INTDisableInterrupts();
        if(p3 != prescale) {                                        // if we are opening the channel or changing the prescaler
            if(T3CONbits.ON) while(TMR3 < PR3 - 64);                // if the timer is running wait until it is near the end of its period
            OpenTimer3(T3_ON | prescale, per);
            p3 = prescale;
        } else if(PR3 != per) {                                     // if we are just changing the frequency
            while(TMR3 > per - 24);                                 // wait for it to be safe to do so
            PR3 = per;
        }
        INTEnableInterrupts();

        // output 2A
        if(!oc1) {
            ExtCfg(PWM_CH4_PIN, EXT_DIG_OUT, 0);                    // this is the first time so initialise everything
            ExtCfg(PWM_CH4_PIN, EXT_COM_RESERVED, 0);
            PWM_CH4_OPEN;
            oc1 = true;
            OC1CON = 0;                                             // reset the OC module so that we can set up again correctly
            OpenOC1( OC_ON | OC_TIMER_MODE16 | OC_TIMER3_SRC | OC_PWM_FAULT_PIN_DISABLE , dcy[0], dcy[0] );
        } else
            OC1RS = dcy[0];                                         // this is output 2A duty cycle

        // output 2B
        if(dcy[1] >= 0) {
            if(!oc4) {
                ExtCfg(PWM_CH5_PIN, EXT_DIG_OUT, 0);                // this is the first time so initialise everything
                ExtCfg(PWM_CH5_PIN, EXT_COM_RESERVED, 0);
                PWM_CH5_OPEN;
                oc4 = true;
                OC4CON = 0;                                         // reset the OC module so that we can set up again correctly
                OpenOC4( OC_ON | OC_TIMER_MODE16 | OC_TIMER3_SRC | OC_PWM_FAULT_PIN_DISABLE , dcy[1], dcy[1] );
            } else
                OC4RS = dcy[1];                                             // this is output 2B duty cycle
        }

        if(dcy[2] >= 0) error("Syntax");


#if defined(MX470)
        CurrentlyPlaying = P_NOTHING;
#endif
    }
}



// close the PWM output
void PWMClose(int channel) {
    switch(channel) {
        case 0:
            if(T2CONbits.ON) while(TMR2 < PR2 - 64);                // wait for the timer to finish its cycle
            T2CONbits.ON = 0;

            if(oc3) {
                CloseOC3();
                PWM_CH1_CLOSE;
                if(ExtCurrentConfig[PWM_CH1_PIN] == EXT_COM_RESERVED) ExtCfg(PWM_CH1_PIN, EXT_NOT_CONFIG, 0);
            }

            if(oc2) {
                CloseOC2();
                PWM_CH2_CLOSE;
                if(ExtCurrentConfig[PWM_CH2_PIN] == EXT_COM_RESERVED) ExtCfg(PWM_CH2_PIN, EXT_NOT_CONFIG, 0);
            }

            if(oc5) {
                CloseOC5();
                PWM_CH3_CLOSE;
                if(ExtCurrentConfig[PWM_CH3_PIN] == EXT_COM_RESERVED) ExtCfg(PWM_CH3_PIN, EXT_NOT_CONFIG, 0);
            }

            oc3 = oc2 = oc5 = false;
            p2 = -1;
            break;
        case 1:
            if(T3CONbits.ON) while(TMR3 < PR3 - 64);                // wait for the timer to finish its cycle
            T3CONbits.ON = 0;

            if(oc1) {
                CloseOC1();
                PWM_CH4_CLOSE;
                if(ExtCurrentConfig[PWM_CH4_PIN] == EXT_COM_RESERVED) ExtCfg(PWM_CH4_PIN, EXT_NOT_CONFIG, 0);
            }

            if(oc4) {
                CloseOC4();
                PWM_CH5_CLOSE;
                if(ExtCurrentConfig[PWM_CH5_PIN] == EXT_COM_RESERVED) ExtCfg(PWM_CH5_PIN, EXT_NOT_CONFIG, 0);
            }

            oc1 = oc4 = false;
            p3 = -1;
            break;
    }
}
